// Content script for YouTube pages
// This script runs on YouTube pages and can interact with the DOM

const LOG_PREFIX_CS = '[CloseYouTube][CS]';
function csInfo(...args){ console.log(LOG_PREFIX_CS, ...args); }
function csWarn(...args){ console.warn(LOG_PREFIX_CS, ...args); }
function csError(...args){ console.error(LOG_PREFIX_CS, ...args); }

function safeSendMessage(message) {
  try {
    if (!chrome || !chrome.runtime || !chrome.runtime.id) return;
    chrome.runtime.sendMessage(message, () => {
      // Swallow errors when the extension reloads/unloads.
      void chrome.runtime.lastError;
    });
  } catch (e) {
    // Extension context invalidated; ignore.
  }
}

csInfo('Content script loaded on:', window.location.href);

// Track time spent on YouTube
let startTime = Date.now();

// Video end monitoring
let videoEndMonitoring = false;
let currentVideo = null;
let videoEndListener = null;

// Initialize content script
function init() {
  setupMessageListener();
  setupThemeWatcher();
  checkAndStartMonitoring();
}



// Listen for messages from background script
function setupMessageListener() {
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    console.log('Content script received message:', message);
    
    if (message.action === 'startVideoEndMonitoring') {
      startVideoEndMonitoring();
    }
    
    if (message.action === 'stopVideoEndMonitoring') {
      stopVideoEndMonitoring();
    }
    
    if (message.action === 'pauseVideo') {
      pauseCurrentVideo();
    }

    if (message.action === 'playIfPaused') {
      playIfPaused();
    }
    
    if (message.action === 'getTimeSpent') {
      const timeSpent = Date.now() - startTime;
      sendResponse({ timeSpent: Math.floor(timeSpent / 60000) });
    }
  });
}

/** Watch OS/Chrome theme and notify background so pinned icon swaps instantly */
function setupThemeWatcher() {
  try {
    const m = window.matchMedia('(prefers-color-scheme: dark)');
    const notify = () => {
      safeSendMessage({
        action: 'updateTheme',
        theme: m.matches ? 'dark' : 'light'
      });
    };
    // initial sync
    notify();
    // listen for changes
    if (typeof m.addEventListener === 'function') {
      m.addEventListener('change', notify);
    } else if (typeof m.addListener === 'function') {
      m.addListener(notify);
    }
  } catch (e) {
    csWarn('Theme watcher setup failed', e);
  }
}

// Check monitoring when navigating within YouTube
let lastUrl = window.location.href;
const urlPoll = setInterval(() => {
  if (window.location.href !== lastUrl) {
    lastUrl = window.location.href;
    startTime = Date.now(); // Reset timer on navigation
    csInfo('URL changed, checking monitoring:', lastUrl);
    checkAndStartMonitoring();
  }
}, 800);

// Video end monitoring functions
let domObserver = null;
function startVideoEndMonitoring() {
  csInfo('Starting video end monitoring');
  videoEndMonitoring = true;
  
  // Find the video element and set up monitoring
  setupVideoMonitoring();
  
  // Also monitor for video changes (YouTube SPA navigation)
  if (domObserver) {
    try { domObserver.disconnect(); } catch (e) { /* ignore */ }
  }
  domObserver = new MutationObserver(() => {
    if (videoEndMonitoring) {
      setupVideoMonitoring();
    }
  });
  domObserver.observe(document.body, {
    childList: true,
    subtree: true
  });
}

function stopVideoEndMonitoring() {
  csInfo('Stopping video end monitoring');
  videoEndMonitoring = false;
  
  if (currentVideo && videoEndListener) {
    currentVideo.removeEventListener('ended', videoEndListener);
    currentVideo = null;
    videoEndListener = null;
  }
  if (domObserver) {
    try { domObserver.disconnect(); } catch (e) { /* ignore */ }
    domObserver = null;
  }
}

function setupVideoMonitoring() {
  // Wait for video element to be available
  const checkForVideo = () => {
    const video = document.querySelector('video');
    
    if (video && video !== currentVideo && videoEndMonitoring) {
      // Remove listener from previous video if exists
      if (currentVideo && videoEndListener) {
        currentVideo.removeEventListener('ended', videoEndListener);
        csInfo('Removed listener from previous video');
      }
      
      currentVideo = video;
      
      // Create new listener
      videoEndListener = () => {
        csInfo('Video ended! Duration:', video.duration, 'Current time:', video.currentTime);
        
        // Double-check that video actually ended (not just paused)
        if (video.ended || (video.currentTime >= video.duration && video.duration > 0)) {
          csInfo('Confirmed video ended, notifying background script');
          
          // Send message to background script
          safeSendMessage({ action: 'videoEnded' });
        }
      };
      
      // Add listener to current video
      currentVideo.addEventListener('ended', videoEndListener);
      csInfo('Video end listener attached');
      csInfo('Video duration:', video.duration, 'Video src:', video.src);
    }
  };
  
  // Check immediately and then periodically
  checkForVideo();
  setTimeout(checkForVideo, 1000);
  setTimeout(checkForVideo, 3000);
  setTimeout(checkForVideo, 5000);
}

/** Pause current video if playing */
function pauseCurrentVideo() {
  const video = document.querySelector('video');
  if (video && !video.paused) {
    video.pause();
    console.log('Video paused by auto-pause feature');
  }
}

/** Play current video if paused */
function playIfPaused() {
  const video = document.querySelector('video');
  if (video && video.paused) {
    video.play().catch(() => {});
    console.log('Video resumed by switch-to-next-tab behavior');
  }
}

// Check for auto-close videos setting on page load and URL changes
function checkAndStartMonitoring() {
  chrome.storage.sync.get(['autoCloseVideos'], (result) => {
    if (result.autoCloseVideos && window.location.href.includes('/watch')) {
      startVideoEndMonitoring();
    } else if (!window.location.href.includes('/watch')) {
      stopVideoEndMonitoring();
    }
  });
}

// Initial check
checkAndStartMonitoring();


// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}
